// Comparing speeds of irr::core vs stl base classes
// TODO: tests for list not finish, string, map and allocaters not yet tested
// TODO: random data should be replaceable by an array of fixed "random" numbers to make it reproducable.
// TODO: instead of TestData a template should be used to make it easier to run it several times with different data


#include <irrlicht.h>
#include <iostream>
#include <vector>
#include <algorithm>
#include <list>
#include <string>
#include <map>
#include <ctime>
#include <fstream>
#include <cstdio>
#include "profiler.h"

#pragma comment(lib, "Irrlicht.lib")

// After each comparison flip is flipped
static bool flip=false;

//! Compare the runtime of two codes
//! init and exit code will be run each outer loop, but the time will not be regarded
//! idA and idB are the profile id's for the code
//! outer_iterations is the number of iterations which should be run including init and exit code
//! inner_iterations is the number of iterations which codeA and codeB will run without initializing
//! codeA and codeB are the codes which should be run
//! The execution of the code will be swapped on each iteration to reduce sequence effects in the timing.
//! The variables OUTER_LOOP and INNER_LOOP can be used within the corresponding loops
#define PROFILE_COMPARE( init, exit, idA, idB, outer_iterations, inner_iterations, codeA, codeB ) \
	for ( int OUTER_LOOP=0; OUTER_LOOP < outer_iterations; ++OUTER_LOOP )\
	{ \
		init \
		for ( int INNER_LOOP=0; INNER_LOOP < inner_iterations; ++INNER_LOOP )\
		{ \
			if ( !flip ) \
			{ \
				flip = true; \
				PROFILE_START(idA); \
				codeA \
				PROFILE_STOP(idA); \
				PROFILE_START(idB); \
				codeB \
				PROFILE_STOP(idB); \
			} \
			else \
			{ \
				flip = false; \
				PROFILE_START(idB); \
				codeB \
				PROFILE_STOP(idB); \
				PROFILE_START(idA); \
				codeA \
				PROFILE_STOP(idA); \
			} \
		} \
		exit \
	}

enum PROFILE_GROUPS
{
	PRGR_ARRAYS = 101,
	PRGR_LISTS,
	PRGR_STRINGS,
	PRGR_MAPS,
};

enum PROFILE_IDS
{
	PRID_MIN = 1,

	// for the arrays
	PRID_ARR_irr_empty_constructor,
	PRID_ARR_std_empty_constructor,

	PRID_ARR_irr_sized_constructor,
	PRID_ARR_std_sized_constructor,

	PRID_ARR_irr_copy_constructor,
	PRID_ARR_std_copy_constructor,

	PRID_ARR_irr_reallocate,
	PRID_ARR_std_reallocate,

	PRID_ARR_irr_push_back,
	PRID_ARR_std_push_back,

	PRID_ARR_irr_push_front,
	PRID_ARR_std_push_front,

	PRID_ARR_irr_insert,
	PRID_ARR_std_insert,

	PRID_ARR_irr_clear,
	PRID_ARR_std_clear,

	PRID_ARR_irr_set_used,
	PRID_ARR_std_set_used,

	PRID_ARR_irr_op_assign,
	PRID_ARR_std_op_assign,

	PRID_ARR_irr_op_equal,
	PRID_ARR_std_op_equal,

	PRID_ARR_irr_op_unequal,
	PRID_ARR_std_op_unequal,

	PRID_ARR_irr_op_access,
	PRID_ARR_std_op_access,

	PRID_ARR_irr_get_last,
	PRID_ARR_std_get_last,

	PRID_ARR_irr_size,
	PRID_ARR_std_size,

	PRID_ARR_irr_allocated_size,
	PRID_ARR_std_allocated_size,

	PRID_ARR_irr_empty,
	PRID_ARR_std_empty,

	PRID_ARR_irr_sort,
	PRID_ARR_std_sort,
	PRID_ARR_irr_sort2,	// same as sort, but want to count it alone
	PRID_ARR_std_sort_heap,

	PRID_ARR_irr_search,
	PRID_ARR_std_search,

	PRID_ARR_irr_binary_search,
	PRID_ARR_std_binary_search,

	PRID_ARR_irr_linear_search,
	PRID_ARR_std_linear_search,

	PRID_ARR_irr_lin_rev_search,
	PRID_ARR_std_lin_rev_search,

	PRID_ARR_irr_erase,
	PRID_ARR_std_erase,

	PRID_ARR_irr_erase_counted,
	PRID_ARR_std_erase_counted,

	// for the lists
	PRID_LST_irr_constr,
	PRID_LST_std_constr,

	PRID_LST_irr_copy_constr,
	PRID_LST_std_copy_constr,

	PRID_LST_irr_op_assign,
	PRID_LST_std_op_assign,

	PRID_LST_irr_size,
	PRID_LST_std_size,

	PRID_LST_irr_clear,
	PRID_LST_std_clear,

	PRID_LST_irr_empty,
	PRID_LST_std_empty,

	PRID_LST_irr_push_back,
	PRID_LST_std_push_back,

	PRID_LST_irr_push_front,
	PRID_LST_std_push_front,

	PRID_LST_irr_begin,
	PRID_LST_std_begin,

	PRID_LST_irr_constbegin,
	PRID_LST_std_constbegin,

	PRID_LST_irr_end,
	PRID_LST_std_end,

	PRID_LST_irr_constend,
	PRID_LST_std_constend,

	PRID_LST_irr_last,
	PRID_LST_std_last,

	PRID_LST_irr_constlast,
	PRID_LST_std_constlast,

	PRID_LST_STD_back,
	PRID_LST_STD_rend,

	PRID_LST_irr_insertafter,
	PRID_LST_std_insertafter,

	PRID_LST_irr_insert,
	PRID_LST_std_insert,

	PRID_LST_irr_erase,
	PRID_LST_std_erase,

	PRID_MAX
};

// Using a class instead of just using integers, so a test if the behaviour is the same
// with more complex types can be done without much work.
class TestData
{
public:
	int value;
	void RandomFill()	{ value = rand(); }

	bool operator==(const TestData & t) const
	{
		return value == t.value;
	}

	bool operator!=(const TestData & t) const
	{
		return value != t.value;
	}

	bool operator<(const TestData & t) const
	{
		return value < t.value;
	}
};

// Create arrays with random size between [sizeFrom_, sizeTo_[
// and fill it with random data.
// This should never be profiled, so it does not matter how long it takes
void fill_random_array(unsigned int minSize_, unsigned int range_, irr::core::array<TestData> & arr_, std::vector<TestData> & vec_ )
{
	if ( !range_ )
		return;

	unsigned int realSize = (rand() % range_) + minSize_;
	arr_.set_used(realSize);
	vec_.resize(realSize);
	for ( unsigned int i=0; i < realSize; ++i )
	{
		TestData data;
		data.RandomFill();
		arr_[i] = data;
		vec_[i] = data;
	}
}

void fill_random_list(unsigned int minSize_, unsigned int range_, irr::core::list<TestData> & irrli_, std::list<TestData> & stlli_)
{
	if ( !range_ )
		return;

	unsigned int realSize = (rand() % range_) + minSize_;
	irrli_.clear();
	stlli_.clear();
	for ( unsigned int i=0; i < realSize; ++i )
	{
		TestData data;
		data.RandomFill();
		irrli_.push_back(data);
		stlli_.push_back(data);
	}
}

void profile_arrays()
{
	// variables needed in many tests
	int count1 = 0;
	int count2 = 0;
	irr::core::array<TestData> arr1;
	irr::core::array<TestData> arr2;
	std::vector<TestData> vec1;
	std::vector<TestData> vec2;
	TestData testDummy;

	printf("constructor\n");
	PROFILE_COMPARE(
		;,
		;,
		PRID_ARR_irr_empty_constructor,
		PRID_ARR_std_empty_constructor,
		1, 100000,
		{
			irr::core::array<TestData> arr;
		}
		,
		{
			std::vector<TestData> vec;
		}
	)

	printf("constructor size\n");
	count1 = 0;
	PROFILE_COMPARE(
		count1 += 10;
		,
		;,
		PRID_ARR_irr_sized_constructor,
		PRID_ARR_std_sized_constructor,
		100,
		200,
		{
			irr::core::array<TestData> arr(count1);
		},
		{
			std::vector<TestData> vec( count1 );
		}
	)

	printf("copy constructor\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		count1 += 10;
		,
		;,
		PRID_ARR_irr_copy_constructor,
		PRID_ARR_std_copy_constructor,
		100,
		200,
		{
			irr::core::array<TestData> arr2(arr);
		},
		{
			std::vector<TestData> vec2( vec );
		}
	)

	// test not really identical as stl is allowed to allocate more (a known weakness)
	printf("reallocate\n");
	count1 = 1;
	count2 = 10000;
	PROFILE_COMPARE(
		count1 += 2;
		--count2;
		,
		;,
		PRID_ARR_irr_reallocate,
		PRID_ARR_std_reallocate,
		500,
		20,
		{
			arr1.reallocate(count1);
			arr2.reallocate(count2);
		},
		{
			vec1.reserve(count1);
			vec2.reserve(count2);
		}
	)

	printf("push_back\n");
	count1 = 10;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		count1 += 10;
		,
		;,
		PRID_ARR_irr_push_back,
		PRID_ARR_std_push_back,
		10,
		10000,
		{
			arr.push_back(testDummy);
		},
		{
			vec.push_back(testDummy);
		}
	)

	printf("push_front\n");
	count1 = 10;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		count1 += 10;
		,
		;,
		PRID_ARR_irr_push_front,
		PRID_ARR_std_push_front,
		10,
		5000,
		{
			arr.push_front(testDummy);
		},
		{
			vec.insert( vec.begin(), testDummy);
		}
	)

	printf("insert\n");
	count1 = 10;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		int pos = rand() % count1;
		count1 += 10;
		,
		;,
		PRID_ARR_irr_insert,
		PRID_ARR_std_insert,
		10,
		5000,
		{
			arr.insert(testDummy, pos);
		},
		{
			vec.insert( vec.begin()+pos, testDummy );
		}
	)

	printf("clear\n");
	count1 = 10;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		count1 += 10;
		,
		;,
		PRID_ARR_irr_clear,
		PRID_ARR_std_clear,
		10,
		10000,
		{
			arr.clear();
		},
		{
			vec.clear();
		}
	)

	printf("set_used\n");
	count1 = 1;
	count2 = 100000;
	PROFILE_COMPARE(
		count1 += 2;
		--count2;
		,
		;,
		PRID_ARR_irr_set_used,
		PRID_ARR_std_set_used,
		500,
		10,
		{
			arr1.set_used(count1);
			arr2.set_used(count2);
		},
		{
			vec1.resize(count1);
			vec2.resize(count2);
		}
	)

	printf("=\n");
	count1 = 10;
	PROFILE_COMPARE(
		fill_random_array(count1, 1, arr1, vec1);
		fill_random_array(count1, 1, arr2, vec2);
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		count1 += 10;
		,
		;,
		PRID_ARR_irr_op_assign,
		PRID_ARR_std_op_assign,
		50,
		100,
		{
			arr = arr1;
			arr1 = arr2;
			arr2 = arr;
		},
		{
			vec = vec1;
			vec1 = vec2;
			vec2 = vec;
		}
	)

	printf("==\n");
	count1 = 10;
	PROFILE_COMPARE(
		fill_random_array(count1, 1, arr1, vec1);
		fill_random_array(count1, 1, arr2, vec2);
		irr::core::array<TestData> arr(arr1);
		std::vector<TestData> vec(vec1);
		count1 += 10;
		,
		;,
		PRID_ARR_irr_op_equal,
		PRID_ARR_std_op_equal,
		50,
		1000,
		{
			bool a = arr1 == arr;
			bool b = arr2 == arr;
		},
		{
			bool a = vec1 == vec;
			bool b = vec2 == vec;
		}
	)

	printf("!=\n");
	count1 = 10;
	PROFILE_COMPARE(
		fill_random_array(count1, 1, arr1, vec1);
		fill_random_array(count1, 1, arr2, vec2);
		irr::core::array<TestData> arr(arr1);
		std::vector<TestData> vec(vec1);
		count1 += 10;
		,
		;,
		PRID_ARR_irr_op_unequal,
		PRID_ARR_std_op_unequal,
		50,
		1000,
		{
			bool a = arr1 != arr;
			bool b = arr2 != arr;
		},
		{
			bool a = vec1 != vec;
			bool b = vec2 != vec;
		}
	)

	printf("[]\n");
	count1 = 10;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		int pos = rand() % count1;
		count1 += 10;
		,
		;,
		PRID_ARR_irr_op_access,
		PRID_ARR_std_op_access,
		10,
		10000,
		{
			TestData & data = arr[pos];
		},
		{
			TestData & data = vec[pos];
		}
	)

	printf("last\n");
	count1 = 10;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		count1 += 10;
		,
		;,
		PRID_ARR_irr_get_last,
		PRID_ARR_std_get_last,
		10,
		10000,
		{
			TestData & data = arr.getLast();
		},
		{
			TestData & data = vec.back();
		}
	)

	printf("size\n");
	count1 = 10;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		count1 += 10;
		,
		;,
		PRID_ARR_irr_size,
		PRID_ARR_std_size,
		10,
		10000,
		{
			irr::u32 s = arr.size();
		},
		{
			std::vector<TestData>::size_type s = vec.size();
		}
	)

	printf("allocated_size\n");
	count1 = 10;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		count1 += 10;
		,
		;,
		PRID_ARR_irr_allocated_size,
		PRID_ARR_std_allocated_size,
		10,
		10000,
		{
			irr::u32 s = arr.allocated_size();
		},
		{
			std::vector<TestData>::size_type s = vec.capacity();
		}
	)

	printf("empty\n");
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;,
		for ( int x=0; x<100;++x)
		{
			arr.push_back(testDummy);
			vec.push_back(testDummy);
		}
		,
		PRID_ARR_irr_empty,
		PRID_ARR_std_empty,
		10,
		10000,
		{
			bool e = arr.empty();
		},
		{
			bool e = vec.empty();
		}
	)

	printf("sort\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		count1 += 1000;
		,
		;,
		PRID_ARR_irr_sort,
		PRID_ARR_std_sort,
		50,
		1,
		{
			arr.set_sorted(false);	// don't know why it doesn't know itself
			arr.sort();
		},
		{
			std::sort( vec.begin(), vec.end() );
		}
	)

	printf("sort_heap\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		count1 += 1000;
		,
		;,
		PRID_ARR_irr_sort2,
		PRID_ARR_std_sort_heap,
		50,
		1,
		{
			arr.set_sorted(false);	// don't know why it doesn't know itself
			arr.sort();
		},
		{
			std::make_heap( vec.begin(), vec.end() );
			std::sort_heap( vec.begin(), vec.end() );
		}
	)

	printf("bin_search\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		arr.sort();
		std::sort( vec.begin(), vec.end() );
		TestData findMe = arr[rand()%count1];
		count1 += 1000;
		,
		;,
		PRID_ARR_irr_search,
		PRID_ARR_std_search,
		10,
		1000,
		{
			arr.binary_search(findMe);
		},
		{
			std::binary_search( vec.begin(), vec.end(), findMe );
		}
	)

	printf("bin_search\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		arr.sort();
		std::sort( vec.begin(), vec.end() );
		TestData findMe = arr[rand()%count1];
		int findStart = rand()%count1;
		int findEnd = rand()%count1;
		if ( findStart > findEnd )
		{
			int h=findEnd;
			findEnd=findStart;
			findStart=h;
		}
		count1 += 1000;
		,
		;,
		PRID_ARR_irr_binary_search,
		PRID_ARR_std_binary_search,
		10,
		10000,
		{
			arr.binary_search(findMe, findStart, findEnd);
		},
		{
			std::binary_search( vec.begin()+findStart, vec.begin()+findEnd, findMe );
		}
	)

	printf("lin_search\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		TestData findMe = arr[rand()%count1];
		count1 += 1000;
		,
		;,
		PRID_ARR_irr_linear_search,
		PRID_ARR_std_linear_search,
		10,
		2000,
		{
			arr.linear_search(findMe);
		},
		{
			std::find( vec.begin(), vec.end(), findMe );
		}
	)

	printf("lin_rev_search\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		TestData findMe = arr[rand()%count1];
		count1 += 1000;
		,
		;,
		PRID_ARR_irr_lin_rev_search,
		PRID_ARR_std_lin_rev_search,
		10,
		10000,
		{
			arr.linear_reverse_search(findMe);
		},
		{
			std::find( vec.rbegin(), vec.rend(), findMe );	// reverse iterators
		}
	)

	printf("erase\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		int eraseStart = rand()%count1;
		int eraseEnd = rand()%count1;
		if ( eraseStart > eraseEnd )
		{
			int h=eraseEnd;
			eraseEnd=eraseStart;
			eraseStart=h;
		}
		count1 += 10;
		,
		;,
		PRID_ARR_irr_erase,
		PRID_ARR_std_erase,
		300,
		1,
		{
			for ( int r=eraseEnd; r >= eraseStart; --r )
			{
				arr.erase((irr::u32)r);
			}
		},
		{
			for ( int r=eraseEnd; r >= eraseStart; --r )
			{
				vec.erase( vec.begin() + r );
			}
		}
	)

	printf("erase counted\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::array<TestData> arr;
		std::vector<TestData> vec;
		fill_random_array(count1, 1, arr, vec);
		int eraseStart = rand()%count1;
		int eraseEnd = rand()%count1;
		if ( eraseStart > eraseEnd )
		{
			int h=eraseEnd;
			eraseEnd=eraseStart;
			eraseStart=h;
		}
		count1 += 10;
		,
		;,
		PRID_ARR_irr_erase_counted,
		PRID_ARR_std_erase_counted,
		5000,
		1,
		{
			arr.erase((irr::u32)eraseStart, (irr::u32)(eraseEnd-eraseStart));
		},
		{
			vec.erase( vec.begin()+eraseStart, vec.begin()+eraseEnd );
		}
	)
}

void profile_lists()
{
	int count1 = 0;
	int count2 = 0;
	irr::core::list<TestData> irrli1;
	irr::core::list<TestData> irrli2;
	std::list<TestData> stlli1;
	std::list<TestData> stlli2;
	TestData testDummy;

	printf("constr\n");
	PROFILE_COMPARE(
		;,
		;,
		PRID_LST_irr_constr,
		PRID_LST_std_constr,
		1, 100000,
		{
			irr::core::list<TestData> irrli;
		}
		,
		{
			std::list<TestData> stlli;
		}
	)

	printf("copy constr\n");
	count1 = 100;
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		fill_random_list(count1, 1, irrli, stlli);
		count1 += 10;
		,
		;,
		PRID_LST_irr_copy_constr,
		PRID_LST_std_copy_constr,
		20, 100,
		{
			irr::core::list<TestData> irrli2(irrli);
		}
		,
		{
			std::list<TestData> stlli2(stlli);
		}
	)

	printf("=\n");
	count1 = 10;
	PROFILE_COMPARE(
		fill_random_list(count1, 1, irrli1, stlli1);
		fill_random_list(count1, 1, irrli2, stlli2);
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		count1 += 10;
		,
		;,
		PRID_LST_irr_op_assign,
		PRID_LST_std_op_assign,
		50, 100,
		{
			irrli = irrli1;
			irrli1 = irrli2;
			irrli2 = irrli;
		}
		,
		{
			stlli = stlli1;
			stlli1 = stlli2;
			stlli2 = stlli;
		}
	)

	count1 = 10;
	printf("size\n");
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		fill_random_list(count1, 1, irrli, stlli);
		count1 += 10;
		,
		;,
		PRID_LST_irr_size,
		PRID_LST_std_size,
		10, 10000,
		{
            irr::u32 s = irrli.getSize();
		}
		,
		{
            std::list<TestData>::size_type s = stlli.size();
		}
	)

	printf("clear\n");
	count1 = 10;
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		fill_random_list(count1, 1, irrli, stlli);
		count1 += 10;
		,
		;,
		PRID_LST_irr_clear,
		PRID_LST_std_clear,
		1000, 2,
		{
		    irrli.clear();
		}
		,
		{
		    stlli.clear();
		}
	)

	printf("empty\n");
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		for ( int x=0; x<100;++x)
		{
			irrli.push_back(testDummy);
			stlli.push_back(testDummy);
		}
		,
		;,
		PRID_LST_irr_empty,
		PRID_LST_std_empty,
		10, 10000,
		{
            bool e = irrli.empty();
		}
		,
		{
            bool e = stlli.empty();
		}
	)

	printf("pushback\n");
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		,
		;,
		PRID_LST_irr_push_back,
		PRID_LST_std_push_back,
		1, 100000,
		{
            irrli.push_back(testDummy);
		}
		,
		{
            stlli.push_back(testDummy);
		}
	)

	printf("pushfront\n");
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		,
		;,
		PRID_LST_irr_push_front,
		PRID_LST_std_push_front,
		1, 100000,
		{
            irrli.push_front(testDummy);
		}
		,
		{
            stlli.push_front(testDummy);
		}
	)

	printf("begin\n");
	count1 = 0;
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		if ( count1 )
		{
		    irrli.push_back(testDummy);
		    stlli.push_back(testDummy);
		}
		++count1;
		,
		;,
		PRID_LST_irr_begin,
		PRID_LST_std_begin,
		2, 100000,
		{
            irr::core::list<TestData>::Iterator it = irrli.begin();
		}
		,
		{
            std::list<TestData>::iterator it = stlli.begin();
		}
	)

	printf("constbegin\n");
	count1 = 0;
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		if ( count1 )
		{
		    irrli.push_back(testDummy);
		    stlli.push_back(testDummy);
		}
		const irr::core::list<TestData> irrliconst(irrli);
		++count1;,
		;,
		PRID_LST_irr_constbegin,
		PRID_LST_std_constbegin,
		2, 100000,
		{
            irr::core::list<TestData>::ConstIterator it = irrliconst.begin();
		}
		,
		{
            std::list<TestData>::const_iterator it = stlli.begin();
		}
	)

	printf("end\n");
	count1 = 1;
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		fill_random_list(count1, 1, irrli, stlli);
		count1 += 10;
		,
		;,
		PRID_LST_irr_end,
		PRID_LST_std_end,
		10, 10000,
		{
            irr::core::list<TestData>::Iterator it = irrli.end();
		}
		,
		{
            std::list<TestData>::iterator it = stlli.end();
		}
	)

	printf("constend\n");
	count1 = 1;
	PROFILE_COMPARE(
		irr::core::list<TestData> irrli;
		std::list<TestData> stlli;
		fill_random_list(count1, 1, irrli, stlli);
		const irr::core::list<TestData> irrliconst(irrli);
		count1 += 10;
		,
		;,
		PRID_LST_irr_constend,
		PRID_LST_std_constend,
		10, 10000,
		{
            irr::core::list<TestData>::ConstIterator it = irrliconst.end();
		}
		,
		{
            std::list<TestData>::const_iterator it = stlli.end();
		}
	)

	printf("last\n");
	count1 = 1;
	PROFILE_COMPARE(
		fill_random_list(count1, 1, irrli1, stlli1);
		count1 += 10;
		,
		;,
		PRID_LST_irr_last,
		PRID_LST_std_last,
		10, 10000,
		{
            irr::core::list<TestData>::Iterator it = irrli1.getLast();
		}
		,
		{
		    if (stlli1.size())
		    {
                std::list<TestData>::iterator it = stlli1.end();
                --it;
		    }
		    else
		    {
                 std::list<TestData>::iterator it = stlli1.begin();
		    }
		}
	)

	printf("constlast\n");
	count1 = 1;
	PROFILE_COMPARE(
		fill_random_list(count1, 1, irrli1, stlli1);
		const irr::core::list<TestData> irrliconst(irrli1);
		count1 += 10;
		,
		;,
		PRID_LST_irr_constlast,
		PRID_LST_std_constlast,
		10, 10000,
		{
            irr::core::list<TestData>::ConstIterator it = irrliconst.getLast();
		}
		,
		{
		    if (stlli1.size())
		    {
                std::list<TestData>::const_iterator it = stlli1.end();
                --it;
            }
            else
            {
                std::list<TestData>::const_iterator it = stlli1.begin();
            }
		}
	)

	// two stl alternatives to access the last element
	printf("stl last alternatives\n");
	count1 = 1;
	PROFILE_COMPARE(
		fill_random_list(count1, 1, irrli1, stlli1);
		fill_random_list(count1, 1, irrli2, stlli2);
		count1 += 10;
		,
		;,
		PRID_LST_STD_back,
		PRID_LST_STD_rend,
		10, 10000,
		{
            TestData & data = stlli1.back();
		}
		,
		{
            std::list<TestData>::reverse_iterator it = stlli1.rbegin();
            ++it;
		}
	)

	printf("insertafter\n");
	count1 = 100;
	PROFILE_COMPARE(
		fill_random_list(count1, 1, irrli1, stlli1);
		int pos = rand() % count1;
		irr::core::list<TestData>::Iterator itirr = irrli1.begin();
		itirr += pos;
        std::list<TestData>::iterator itstl = stlli1.begin();
        for ( int i=0;i<pos;++i)
            ++itstl;    // list has no random access operator
		count1 += 10;
		,
		;,
		PRID_LST_irr_insertafter,
		PRID_LST_std_insertafter,
		50, 1000,
		{
            irrli1.insert_after(itirr, testDummy);
		}
		,
		{
            if ( itstl != stlli1.end() )
            {
                ++itstl;
                stlli1.insert( itstl, testDummy );
            }
            else
                stlli1.push_back(testDummy);
		}
	)

	printf("insert\n");
	count1 = 100;
	PROFILE_COMPARE(
		fill_random_list(count1, 1, irrli1, stlli1);
		int pos = rand() % count1;
		irr::core::list<TestData>::Iterator itirr = irrli1.begin() + pos;
        std::list<TestData>::iterator itstl = stlli1.begin();
        for ( int i=0;i<pos;++i)
            ++itstl;    // list has no random access operator
		count1 += 10;
		,
		;,
		PRID_LST_irr_insert,
		PRID_LST_std_insert,
		50, 1000,
		{
		    irrli1.insert_before(itirr, testDummy);
		}
		,
		{
		    stlli1.insert( itstl, testDummy );
		}
	)

	printf("erase\n");
	count1 = 100;
	PROFILE_COMPARE(
        fill_random_list(count1, 1, irrli1, stlli1);
		int first = rand()%count1;
		irr::core::list<TestData>::Iterator itirr = irrli1.begin()+first;
        std::list<TestData>::iterator itstl = stlli1.begin();
        for ( int i=0;i<first;++i)
            ++itstl;    // list has no random access operator
        int numberruns = rand()%(count1-first-1);
		count1 += 10;
		,
        ;,
		PRID_LST_irr_erase,
		PRID_LST_std_erase,
		500, numberruns,
		{
		    itirr = irrli1.erase(itirr);
		}
		,
		{
            itstl = stlli1.erase(itstl);
		}
	)
}

int main()
{
	srand((unsigned int)time(0));
	irr::video::E_DRIVER_TYPE driverType = irr::video::EDT_SOFTWARE;
	irr::IrrlichtDevice *device =
		irr::createDevice(driverType, irr::core::dimension2d<irr::s32>(640, 480), 16, false);
	if (!device)
		return 1; // could not create selected driver.

	std::ofstream streamProfile( "profile.txt" );
	irr::gPROFILER.Init(device->getTimer(), device->getGUIEnvironment());

    PROFILE_ADD_GROUP( PRGR_ARRAYS, "arrays");
	PROFILE_ADD( PRID_ARR_irr_empty_constructor, PRGR_ARRAYS, "ARR_irr_empty_constr");
	PROFILE_ADD( PRID_ARR_std_empty_constructor, PRGR_ARRAYS, "ARR_std_empty_constr");
	PROFILE_ADD( PRID_ARR_irr_sized_constructor, PRGR_ARRAYS, "ARR_irr_sized_constr");
	PROFILE_ADD( PRID_ARR_std_sized_constructor, PRGR_ARRAYS, "ARR_std_sized_constr");
	PROFILE_ADD( PRID_ARR_irr_copy_constructor, PRGR_ARRAYS, "ARR_irr_copy_constr");
	PROFILE_ADD( PRID_ARR_std_copy_constructor, PRGR_ARRAYS, "ARR_std_copy_constr");
	PROFILE_ADD( PRID_ARR_irr_reallocate, PRGR_ARRAYS, "ARR_irr_reallocate");
	PROFILE_ADD( PRID_ARR_std_reallocate, PRGR_ARRAYS, "ARR_std_reallocate");
	PROFILE_ADD( PRID_ARR_irr_push_back, PRGR_ARRAYS, "ARR_irr_push_back");
	PROFILE_ADD( PRID_ARR_std_push_back, PRGR_ARRAYS, "ARR_std_push_back");
	PROFILE_ADD( PRID_ARR_irr_push_front, PRGR_ARRAYS, "ARR_irr_push_front");
	PROFILE_ADD( PRID_ARR_std_push_front, PRGR_ARRAYS, "ARR_std_push_front");
	PROFILE_ADD( PRID_ARR_irr_insert, PRGR_ARRAYS, "ARR_irr_insert");
	PROFILE_ADD( PRID_ARR_std_insert, PRGR_ARRAYS, "ARR_std_insert");
	PROFILE_ADD( PRID_ARR_irr_clear, PRGR_ARRAYS, "ARR_irr_clear");
	PROFILE_ADD( PRID_ARR_std_clear, PRGR_ARRAYS, "ARR_std_clear");
	PROFILE_ADD( PRID_ARR_irr_set_used, PRGR_ARRAYS, "ARR_irr_set_used");
	PROFILE_ADD( PRID_ARR_std_set_used, PRGR_ARRAYS, "ARR_std_set_used");
	PROFILE_ADD( PRID_ARR_irr_op_assign, PRGR_ARRAYS, "ARR_irr_op_assign");
	PROFILE_ADD( PRID_ARR_std_op_assign, PRGR_ARRAYS, "ARR_std_op_assign");
	PROFILE_ADD( PRID_ARR_irr_op_equal, PRGR_ARRAYS, "ARR_irr_op_equal");
	PROFILE_ADD( PRID_ARR_std_op_equal, PRGR_ARRAYS, "ARR_std_op_equal");
	PROFILE_ADD( PRID_ARR_irr_op_unequal, PRGR_ARRAYS, "ARR_irr_op_unequal");
	PROFILE_ADD( PRID_ARR_std_op_unequal, PRGR_ARRAYS, "ARR_std_op_unequal");
	PROFILE_ADD( PRID_ARR_irr_op_access, PRGR_ARRAYS, "ARR_irr_op_access");
	PROFILE_ADD( PRID_ARR_std_op_access, PRGR_ARRAYS, "ARR_std_op_access");
	PROFILE_ADD( PRID_ARR_irr_get_last, PRGR_ARRAYS, "ARR_irr_get_last");
	PROFILE_ADD( PRID_ARR_std_get_last, PRGR_ARRAYS, "ARR_std_get_last");
	PROFILE_ADD( PRID_ARR_irr_size, PRGR_ARRAYS, "ARR_irr_size");
	PROFILE_ADD( PRID_ARR_std_size, PRGR_ARRAYS, "ARR_std_size");
	PROFILE_ADD( PRID_ARR_irr_allocated_size, PRGR_ARRAYS, "ARR_irr_alloc_size");
	PROFILE_ADD( PRID_ARR_std_allocated_size, PRGR_ARRAYS, "ARR_std_alloc_size");
	PROFILE_ADD( PRID_ARR_irr_empty, PRGR_ARRAYS, "ARR_irr_empty");
	PROFILE_ADD( PRID_ARR_std_empty, PRGR_ARRAYS, "ARR_std_empty");
	PROFILE_ADD( PRID_ARR_irr_sort, PRGR_ARRAYS, "ARR_irr_sort");
	PROFILE_ADD( PRID_ARR_std_sort, PRGR_ARRAYS, "ARR_std_sort");
	PROFILE_ADD( PRID_ARR_irr_sort2, PRGR_ARRAYS, "ARR_irr_sort2");
	PROFILE_ADD( PRID_ARR_std_sort_heap, PRGR_ARRAYS,"ARR_std_sort_heap");
	PROFILE_ADD( PRID_ARR_irr_search, PRGR_ARRAYS, "ARR_irr_search");
	PROFILE_ADD( PRID_ARR_std_search, PRGR_ARRAYS, "ARR_std_search");
	PROFILE_ADD( PRID_ARR_irr_binary_search, PRGR_ARRAYS, "ARR_irr_bin_search");
	PROFILE_ADD( PRID_ARR_std_binary_search, PRGR_ARRAYS, "ARR_std_bin_search");
	PROFILE_ADD( PRID_ARR_irr_linear_search, PRGR_ARRAYS, "ARR_irr_lin_search");
	PROFILE_ADD( PRID_ARR_std_linear_search, PRGR_ARRAYS, "ARR_std_lin_search");
	PROFILE_ADD( PRID_ARR_irr_lin_rev_search, PRGR_ARRAYS, "ARR_irr_lrev_search");
	PROFILE_ADD( PRID_ARR_std_lin_rev_search, PRGR_ARRAYS, "ARR_std_lrev_search");
	PROFILE_ADD( PRID_ARR_irr_erase, PRGR_ARRAYS, "ARR_irr_erase");
	PROFILE_ADD( PRID_ARR_std_erase, PRGR_ARRAYS, "ARR_std_erase");
	PROFILE_ADD( PRID_ARR_irr_erase_counted, PRGR_ARRAYS, "ARR_irr_erase2");
	PROFILE_ADD( PRID_ARR_std_erase_counted, PRGR_ARRAYS, "ARR_std_erase2");

	PROFILE_ADD_GROUP( PRGR_LISTS, "lists");
	PROFILE_ADD( PRID_LST_irr_constr, PRGR_LISTS, "LST_irr_constr");
	PROFILE_ADD( PRID_LST_std_constr, PRGR_LISTS, "LST_std_constr");
	PROFILE_ADD( PRID_LST_irr_copy_constr, PRGR_LISTS, "LST_irr_copy_constr");
	PROFILE_ADD( PRID_LST_std_copy_constr, PRGR_LISTS, "LST_std_copy_constr");
	PROFILE_ADD( PRID_LST_irr_op_assign, PRGR_LISTS, "LST_irr_op_assign");
	PROFILE_ADD( PRID_LST_std_op_assign, PRGR_LISTS, "LST_std_op_assign");
	PROFILE_ADD( PRID_LST_irr_size, PRGR_LISTS, "LST_irr_size");
	PROFILE_ADD( PRID_LST_std_size, PRGR_LISTS, "LST_std_size");
	PROFILE_ADD( PRID_LST_irr_clear, PRGR_LISTS, "LST_irr_clear");
	PROFILE_ADD( PRID_LST_std_clear, PRGR_LISTS, "LST_std_clear");
	PROFILE_ADD( PRID_LST_irr_empty, PRGR_LISTS, "LST_irr_empty");
	PROFILE_ADD( PRID_LST_std_empty, PRGR_LISTS, "LST_std_empty");
	PROFILE_ADD( PRID_LST_irr_push_back, PRGR_LISTS, "LST_irr_push_back");
	PROFILE_ADD( PRID_LST_std_push_back, PRGR_LISTS, "LST_std_push_back");
	PROFILE_ADD( PRID_LST_irr_push_front, PRGR_LISTS, "LST_irr_push_front");
	PROFILE_ADD( PRID_LST_std_push_front, PRGR_LISTS, "LST_std_push_front");
	PROFILE_ADD( PRID_LST_irr_begin, PRGR_LISTS, "LST_irr_begin");
	PROFILE_ADD( PRID_LST_std_begin, PRGR_LISTS, "LST_std_begin");
	PROFILE_ADD( PRID_LST_irr_constbegin, PRGR_LISTS, "LST_irr_constbegin");
	PROFILE_ADD( PRID_LST_std_constbegin, PRGR_LISTS, "LST_std_constbegin");
	PROFILE_ADD( PRID_LST_irr_end, PRGR_LISTS, "LST_irr_end");
	PROFILE_ADD( PRID_LST_std_end, PRGR_LISTS, "LST_std_end");
	PROFILE_ADD( PRID_LST_irr_constend, PRGR_LISTS, "LST_irr_constend");
	PROFILE_ADD( PRID_LST_std_constend, PRGR_LISTS, "LST_std_constend");
	PROFILE_ADD( PRID_LST_irr_last, PRGR_LISTS, "LST_irr_last");
	PROFILE_ADD( PRID_LST_std_last, PRGR_LISTS, "LST_std_last");
	PROFILE_ADD( PRID_LST_irr_constlast, PRGR_LISTS, "LST_irr_constlast");
	PROFILE_ADD( PRID_LST_std_constlast, PRGR_LISTS, "LST_std_constlast");
	PROFILE_ADD( PRID_LST_STD_back, PRGR_LISTS, "LST_STD_back");
	PROFILE_ADD( PRID_LST_STD_rend, PRGR_LISTS, "LST_STD_rend");
	PROFILE_ADD( PRID_LST_irr_insertafter, PRGR_LISTS, "LST_irr_insertafter");
	PROFILE_ADD( PRID_LST_std_insertafter, PRGR_LISTS, "LST_std_insertafter");
	PROFILE_ADD( PRID_LST_irr_insert, PRGR_LISTS, "LST_irr_insert");
	PROFILE_ADD( PRID_LST_std_insert, PRGR_LISTS, "LST_std_insert");
	PROFILE_ADD( PRID_LST_irr_erase, PRGR_LISTS, "LST_irr_erase");
	PROFILE_ADD( PRID_LST_std_erase, PRGR_LISTS, "LST_std_erase");

	PROFILE_ADD_GROUP( PRGR_STRINGS, "strings");
	PROFILE_ADD_GROUP( PRGR_MAPS, "maps");

	profile_arrays();
	irr::gPROFILER.PrintAll(streamProfile);
	irr::gPROFILER.ResetAll();

	profile_lists();
	irr::gPROFILER.PrintAll(streamProfile);
	irr::gPROFILER.ResetAll();

	device->drop();

	return 0;
}

